﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.EventSystems;

[RequireComponent(typeof(Button))]
public class CustomButton : MonoBehaviour, ISelectHandler, IDeselectHandler, IPointerEnterHandler, IPointerExitHandler
{
    public bool IsBackButton;   //Makes the button play the back sound, does NOT automatically navigate

    private Button _Button { get { return GetComponent<Button>(); } }
    private AudioSource _AudioSource { get { return GetComponent<AudioSource>(); } }

    private Animator _Animator;

    void Awake()
    {
        //Get the components, subscribe to events
        gameObject.AddComponent<AudioSource>();
        _AudioSource.playOnAwake = false;
        _Button.onClick.AddListener(OnSelected);
        _Animator = _Button.GetComponent<Animator>();
        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (!e.Handled)
        {
            if (e.NewState == Constants.InputState.MouseAndKeyboard || PageManager.CurrentPage == null || PageManager.CurrentPage.DefaultFocusedControl == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject != gameObject)
            {
                //We're using mouse and keyboard, don't have a page or aren't the focused control so we've lost focus
                OnLostFocus();
            }

            else
            {
                if (e.NewState == Constants.InputState.Controller)
                {
                    //We're using a controller and must have focus so call OnFocus
                    OnFocus();
                }
            }
        }
    }

    private void OnFocus()
    {
        PlayFocusAudio();   //Play the audio

        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnFocus");
            _Animator.ResetTrigger("OnLostFocus");
            _Animator.SetTrigger("OnFocus");
        }
    }

    private void OnLostFocus()
    {
        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnFocus");
            _Animator.ResetTrigger("OnLostFocus");
            _Animator.SetTrigger("OnLostFocus");
        }
    }

    public void OnSelect(BaseEventData eventData)
    {
        if (InputManager.Instance.InputState == Constants.InputState.Controller)    //So we don't play it twice when we're using mouse
        {
            OnFocus();
        }
    }

    public void OnDeselect(BaseEventData data)
    {
        OnLostFocus();
    }

    public void OnPointerEnter(PointerEventData eventData)
    {
        OnFocus();
    }

    public void OnPointerExit(PointerEventData data)
    {
        OnLostFocus();
    }

    private void OnSelected()
    {
        OnLostFocus();
        PlaySelectedAudio();
    }

    private void OnDestroy()
    {
        //Unsubscribe from events
        if (InputManager.Instance != null)
        {
            InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        }
    }

    private void PlayFocusAudio()
    {
        if (AudioManager.Instance != null)
        {
            //We have an audio manager
            if (_Button.IsInteractable())
            {
                //We can interact, play normal focus audio
                AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusClip);
            }

            else
            {
                //We can't interact, play disabled focus audio
                AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusDisabledClip);
            }
        }
    }

    private void PlaySelectedAudio()
    {
        if (AudioManager.Instance != null)
        {
            //We have an audio manager
            if (IsBackButton)
            {
                //We're a back button, play the back sound
                AudioManager.Instance.PlaySFX(AudioManager.Instance.UIBackClip);
            }

            else
            {
                //We're a normal button, play the select sound
                AudioManager.Instance.PlaySFX(AudioManager.Instance.UISelectClip);
            }
        }
    }
}
